package org.msh.tb.bd.dashboard.summaryrep;

import org.jboss.seam.Component;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.*;
import org.jboss.seam.annotations.Transactional;
import org.jboss.seam.annotations.async.Asynchronous;
import org.jboss.seam.annotations.async.Expiration;
import org.jboss.seam.annotations.async.IntervalCron;
import org.jboss.seam.async.QuartzTriggerHandle;
import org.msh.tb.application.SystemErrorDispatcher;
import org.msh.tb.bd.dashboard.DashboardIndicatorUtils;
import org.msh.tb.bd.dashboard.query.*;
import org.msh.tb.bd.dashboard.summaryrep.data.GeoDistributionData;
import org.msh.tb.bd.dashboard.summaryrep.data.SummaryIndicatorData;
import org.msh.tb.bd.dashboard.summaryrep.data.SummaryIndicatorsData;
import org.msh.tb.bd.dashboard.summaryrep.parser.*;
import org.msh.tb.entities.*;
import org.msh.tb.taskscheduling.ScheduledTask;
import org.msh.utils.DbCacheService;
import org.msh.utils.date.DateUtils;

import javax.persistence.EntityManager;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * Created by mauri on 29/05/2017.
 * Generates the @{@link SummaryIndicatorsData} object based on database information.
 */
@Name("summaryIndicatorsGenerator")
@AutoCreate
@Scope(ScopeType.APPLICATION)
public class SummaryIndicatorsGenerator extends ScheduledTask {

    private final String EVENT_NAME = "Dashboard Summarized Indicators Update";
    private final String EVENT_DESC = "Updated the indicators database cache displayed on dashboard screen.";

    @In(create=true)
    DbCacheService dbCacheService;

    @In
    EntityManager entityManager;

    /**
     * Links an IndicatorParser to an IndicatorQuery instance
     */
    private SummaryIndicatorsGenerator.IndicatorDef[] defs = new SummaryIndicatorsGenerator.IndicatorDef[]{
            new SummaryIndicatorsGenerator.IndicatorDef(new Indicator01Query(), new PopulationIndicatorParser("indicator01")),
            new SummaryIndicatorsGenerator.IndicatorDef(new Indicator02Query(), new PopulationIndicatorParser("indicator02")),
            new SummaryIndicatorsGenerator.IndicatorDef(new Indicator03Query(), new PopulationIndicatorParser("indicator03")),
            new SummaryIndicatorsGenerator.IndicatorDef(new Indicator04Query(), new PercentageIndicatorParser("indicator04")),
            new SummaryIndicatorsGenerator.IndicatorDef(new Indicator05Query(), new PercentageIndicatorParser("indicator05")),
            new SummaryIndicatorsGenerator.IndicatorDef(new Indicator06Query(), new Indicator06Parser("indicator06")),
            new SummaryIndicatorsGenerator.IndicatorDef(new Indicator07Query(), new Indicator07Parser("indicator07")),
            new SummaryIndicatorsGenerator.IndicatorDef(new Indicator08Query(), new Indicator08Parser("indicator08")),
            new SummaryIndicatorsGenerator.IndicatorDef(new Indicator09Query(), new Indicator09Parser("indicator09")),
            new SummaryIndicatorsGenerator.IndicatorDef(new Indicator10Query(), new PercentageIndicatorParser("indicator10")),
            new SummaryIndicatorsGenerator.IndicatorDef(new Indicator11Query(), new Indicator11Parser("indicator11"))
    };

    /**
     * Generates/Updates the dashboard report cache based on the user workspaces.
     * It will only be generated/updated for Workspace 11 (production one).
     */
    @Transactional
    public void generateIndicators() {
        try {
            // get workspace
            // will generate only to production workspace to keep dbcache table only with useful cache
            Workspace workspace = entityManager.find(Workspace.class, 11);

            // get national results
            List<SummaryIndicatorData> nationalResults = updateNationalIndicators(workspace);

            // find user workspaces
            List<UserWorkspace> userWorkspaces = entityManager.createQuery("from UserWorkspace uw where uw.workspace.id = :wid")
                    .setParameter("wid", workspace.getId())
                    .getResultList();

            // store list of already updated administrative units
            List<Integer> alreadyUpdated = new ArrayList<Integer>();

            for (UserWorkspace uw : userWorkspaces) {

                AdministrativeUnit adminUnit = DashboardIndicatorUtils.getAdminUnit(uw);

                /* update only if adminUnit is not null, what means that users with national
                view was already updated before, when updating the workspace view.
                and if admin unit was not updated before by this thread */
                if (adminUnit != null && !alreadyUpdated.contains(adminUnit.getId())) {
                    updateIndicators(adminUnit, nationalResults);
                    alreadyUpdated.add(adminUnit.getId());
                }
            }

            /*saveTransactionLog(null, 11);*/
        } catch (Exception e) {
            e.printStackTrace();
            SystemErrorDispatcher systemErrorDispatcher = (SystemErrorDispatcher)Component.getInstance("systemErrorDispatcher");
            systemErrorDispatcher.dispatch(e, null, "others/restricted/dashboardind.seam", null);
        }
    }

    /**
     * Generates/Updates the dashboard report cache for a single user workspace
     */
    public void generateIndicators(UserWorkspace userWorkspace) {

        List<SummaryIndicatorData> nationalResults = updateNationalIndicators(userWorkspace.getWorkspace());

        AdministrativeUnit adminUnit = DashboardIndicatorUtils.getAdminUnit(userWorkspace);

        if (adminUnit == null) {
            // has national view, indicator was already generated
            return;
        }

        updateIndicators(adminUnit, nationalResults);
    }

    /**
     * Generates/Updates the workspace level cached indicators
     * @param workspace
     */
    private List<SummaryIndicatorData> updateNationalIndicators(Workspace workspace) {

        List<SummaryIndicatorData> indicatorList = new ArrayList<SummaryIndicatorData>();

        // generate national indicators
        for (SummaryIndicatorsGenerator.IndicatorDef def : defs) {
            IndicatorQuery query = def.getQuery();
            IndicatorParser parser = def.getParser();

            List<Object[]> r;

            if (query != null) {
                query.setEntityManager(entityManager);
                query.setWorkspace(workspace);
                query.setIniDate(parser.getPeriod().getIniDate());
                query.setEndDate(parser.getPeriod().getEndDate());
                query.setSelectedAdminUnit(null);

                r = query.getSummaryResult();
            } else {
                r = null;
            }

            SummaryIndicatorData data = parser.parse(r, workspace.getName().getName1());

            indicatorList.add(data);
        }


        // create wrapper data object
        SummaryIndicatorsData cacheData = new SummaryIndicatorsData();
        cacheData.setGenerationDate(DateUtils.getDate());
        cacheData.setIndicators(indicatorList);

        // generate and set geo distribution data
        cacheData.setGeoDistributionData(generateGeoDistributionData(workspace, null));

        // generate national hash
        String hash = DashboardIndicatorUtils.getHash(workspace);

        // save cache
        dbCacheService.save(hash, cacheData);

        return indicatorList;
    }

    /**
     * Generates/Updates the indicators for a single administrative unit
     * @param adminUnit
     * @param nationalResult used as comparison rate when parsing
     */
    private void updateIndicators(AdministrativeUnit adminUnit, List<SummaryIndicatorData> nationalResult) {

        if (adminUnit == null) {
            throw new RuntimeException("Admin Unit param must not be null");
        }

        String adminUnitDesc = adminUnit.getCountryStructure().getName().getName1() + ": " + adminUnit.getName().getName1();

        List<SummaryIndicatorData> indicatorList = new ArrayList<SummaryIndicatorData>();

        // generate indicators
        int i = 0;
        for (SummaryIndicatorsGenerator.IndicatorDef def : defs) {
            IndicatorQuery query = def.getQuery();
            IndicatorParser parser = def.getParser();

            List<Object[]> r;

            if (query != null) {
                query.setEntityManager(entityManager);
                query.setWorkspace(adminUnit.getWorkspace());
                query.setIniDate(parser.getPeriod().getIniDate());
                query.setEndDate(parser.getPeriod().getEndDate());
                query.setSelectedAdminUnit(adminUnit);

                r = query.getSummaryResult();
            } else {
                r = null;
            }

            SummaryIndicatorData data = parser.parse(r, adminUnitDesc, nationalResult.get(i));

            indicatorList.add(data);
            i++;
        }

        // create wrapper data object
        SummaryIndicatorsData cacheData = new SummaryIndicatorsData();
        cacheData.setGenerationDate(DateUtils.getDate());
        cacheData.setIndicators(indicatorList);

        // generate and set geo distribution data
        cacheData.setGeoDistributionData(generateGeoDistributionData(adminUnit.getWorkspace(), adminUnit));

        // generate national hash
        String hash = DashboardIndicatorUtils.getHash(adminUnit);

        // save cache
        dbCacheService.save(hash, cacheData);
    }

    private GeoDistributionData generateGeoDistributionData(Workspace workspace, AdministrativeUnit administrativeUnit){

        //generates only for workspace, division and districts view
        if (administrativeUnit != null && administrativeUnit.getLevel() >= 3) {
            return null;
        }

        GeoDistributionQuery query = new GeoDistributionQuery();
        GeoDistributionParser parser = new GeoDistributionParser();

        query.setEntityManager(entityManager);
        query.setWorkspace(workspace);
        query.setIniDate(parser.getPeriod().getIniDate());
        query.setEndDate(parser.getPeriod().getEndDate());
        query.setSelectedAdminUnit(administrativeUnit);

        List<Object[]> r = query.getSummaryResult();
        String locationDesc = administrativeUnit == null ? workspace.getName().getName1() : administrativeUnit.getName().getName1();

        GeoDistributionData data = parser.parse(r, locationDesc);
        data.setCode("geodistribution");

        data.setMaximumCircleSize(DashboardIndicatorUtils.getMaximumCircleSize(administrativeUnit));

        return data;
    }

    @Override
    @Asynchronous
    @Transactional
    public QuartzTriggerHandle createQuartzTimer(
            @Expiration Date when, @IntervalCron String interval) {
        generateIndicators();
        return null;
    }

    @Override
    protected String getEventName() {
        return EVENT_NAME;
    }

    @Override
    protected String getLogDescription() {
        return EVENT_DESC;
    }

    private class IndicatorDef {
        private IndicatorQuery query;
        private IndicatorParser parser;

        public IndicatorDef(IndicatorQuery query, IndicatorParser parser) {
            this.query = query;
            this.parser = parser;
        }

        public IndicatorQuery getQuery() {
            return query;
        }

        public void setQuery(IndicatorQuery query) {
            this.query = query;
        }

        public IndicatorParser getParser() {
            return parser;
        }

        public void setParser(IndicatorParser parser) {
            this.parser = parser;
        }
    }

}
